# run_scoringanimations4.py
import os, subprocess, shlex, tempfile, json

AFTER_EFFECTS_APP = "Adobe After Effects 2025"

# Paths
PROJECT   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/ScoringAnimations4.aep"
CSV_PATH  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
# JSX_BATCH = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_ScoringAnimations4.jsx"
JSX_BATCH = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_ScoringAnimations4-CFB.jsx"

# AE / template settings
COMP        = "Comp-TD-192-CFB"
ANIM_NAME   = "_TD_3_3"
RS_TEMPLATE = "Best Settings"
OM_TEMPLATE = "PNG Sequence"

# Output
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/TD_Animation_4_192"
PATH_TEMPLATE = "{league}"
PURGE_BEFORE  = "1"
NO_RENDER     = "0"
QUIT_APP      = "1"

# League-only
LEAGUE = "CFB"

def _write_bridge_jsx(env: dict) -> str:
    # Write a JSON backstop so JSX can read settings even if $.getenv is empty
    cfg_fd, cfg_path = tempfile.mkstemp(prefix="ae_cfg_", suffix=".json")
    os.close(cfg_fd)
    with open(cfg_path, "w") as f:
        json.dump(env, f)

    def esc(s): return str(s).replace("\\","\\\\").replace('"','\\"')
    lines = []
    for k, v in env.items():
        lines.append('$.setenv("'+esc(k)+'","'+esc(v)+'");')
    lines.append('$.setenv("AE_BRIDGE_JSON","'+esc(cfg_path)+'");')
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')

    code = "\n".join(lines) + "\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    return tmp.name

def run(env: dict):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa = [
        "osascript",
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to activate',
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to DoScriptFile POSIX file "{bridge}"',
    ]
    print("Running:", " ".join(shlex.quote(c) for c in osa))
    subprocess.run(osa, check=True)

def make_env() -> dict:
    return {
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,

        # Layer names
        "AE_TEAMNAME_LAYER": "TeamName",
        "AE_SOLID_LAYER": "Solid",
        "AE_ART_PRECOMP_LAYER": "ART_PRECOMP",
        "AE_SPIKEMAN_LAYER": "SpikeMan",
        "AE_TOUCHDOWN_LAYER": "TOUCHDOWN",

        # Logo configuration
        "AE_LOGO_LAYER": "TeamLogo",
        "AE_LOGO_DIR": "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large",
        "AE_LOGO_PATH_TEMPLATE": "{league}/{abbr}",
        "AE_LOGO_EXTS": "png,jpg,jpeg,svg,ai,psd",
        "AE_LOGO_CASE": "keep",

        # Output configuration (PNG sequence)
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_ANIM": ANIM_NAME,
        "AE_RS_TEMPLATE": RS_TEMPLATE,
        "AE_OM_TEMPLATE": OM_TEMPLATE,

        "AE_PURGE_BEFORE_RENDER": PURGE_BEFORE,
        "AE_NO_RENDER": NO_RENDER,
        "AE_QUIT": QUIT_APP,

        # League-only
        "AE_LEAGUE": LEAGUE,
    }

if __name__ == "__main__":
    assert os.path.isfile(JSX_BATCH), f"JSX not found: {JSX_BATCH}"
    assert os.path.isfile(PROJECT),   f"Project not found: {PROJECT}"
    assert os.path.isfile(CSV_PATH),  f"CSV not found: {CSV_PATH}"
    run(make_env())
